package com.akamai.ampdemo;

import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.os.Bundle;
import android.view.View;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;

import com.akamai.amp.media.VideoPlayerContainer;
import com.akamai.amp.media.VideoPlayerView;
import com.akamai.amp.media.elements.MediaResource;
import com.akamai.amp.media.elements.QualityLevel;
import com.akamai.amp.media.errors.ErrorType;
import com.akamai.amp.uimobile.generic.listeners.IMediaPlayerControllerListener;
import com.akamai.amp.uimobile.generic.media.PlayerControlsOverlay;
import com.akamai.amp.utils.LogManager;


public class MainActivity extends AppCompatActivity {

    private static final String LOG_TAG = "AMP + generic UI";
    //private static final String VIDEO_URL = "https://multiplatform-f.akamaihd.net/i/multi/will/bunny/big_buck_bunny_,640x360_400,640x360_700,640x360_1000,950x540_1500,.f4v.csmil/master.m3u8";
    //private static final String VIDEO_URL = "https://commondatastorage.googleapis.com/gtv-videos-bucket/sample/BigBuckBunny.mp4";
    private static final String VIDEO_URL = "https://media-demo-host.akamaized.net/test/video/kluge/hls/all.m3u8";

    private VideoPlayerContainer videoPlayerContainer;
    private VideoPlayerView videoPlayerView;
    private PlayerControlsOverlay mPlayerControlsOverlay;
    private LinearLayout bottomContainer;
    private ProgressBar progressBar;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        progressBar = findViewById(R.id.progress_bar);
        bottomContainer = findViewById(R.id.bottom_container);
        mPlayerControlsOverlay = findViewById(R.id.playerControls);
        videoPlayerContainer = findViewById(R.id.video_player_container);

        //It configures the controls overlay the first time
        setInlineUI();
        /*
        The controls overlay has it own listener and a list of possible events, through this interface
        is how you get notified when the controls handled internally are interacting with the user
        */
        mPlayerControlsOverlay.addEventsListener(getMediaPlayerControllerListener());

        /*
        enableLongPress(true) enables the showing of controls in screen by long
        screen pressing instead of screen simple screen tap.
        */
        //mPlayerControlsOverlay.enableLongPress(true);

        videoPlayerContainer.addVideoPlayerContainerCallback(getVideoPlayerContainerCallback());
        videoPlayerContainer.setApiKey("demo");
        videoPlayerContainer.prepareResource(VIDEO_URL);


        Button buttonLogQualityLevel = findViewById(R.id.button_quality_level);
        buttonLogQualityLevel.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view){
                if (videoPlayerView != null) {
                    QualityLevel qualityLevel = videoPlayerView.getSelectedQualityLevel();
                    Integer index = videoPlayerView.getSelectedQualityLevelIndex();
                    String message = buildMessage(qualityLevel, index);
                    LogManager.log(LOG_TAG, message);
                    Toast.makeText(getApplicationContext(), message, Toast.LENGTH_SHORT).show();
                }
            }
        });

    }

    private String buildMessage(QualityLevel qualityLevel, Integer index) {
        String message = "";
        if (qualityLevel != null){
            message += qualityLevel;
        }
        if (index != null) {
            if (index == -1)
            {
                message += "Adaptive Bitrate";
            }
            else
            {
                message += " Index: " + index;
            }
        }
        return message;
    }

    @Override
    public void onResume() {
        if (videoPlayerView != null) {
            videoPlayerView.onResume();
        }
        super.onResume();
    }

    @Override
    public void onPause() {
        if(videoPlayerView != null){
            videoPlayerView.onPause();
        }
        super.onPause();
    }

    @Override
    protected void onDestroy() {
        if (videoPlayerView != null) {
            videoPlayerView.stop();
            videoPlayerView.onDestroy();
        }
        super.onDestroy();
    }

    //To control when the activity rotates, the onConfigurationChanged is implemented
    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        // Checks the orientation of the screen
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            //It overrides the Control Overlay with the FullScreen controls
            setFullScreenUI();
            /*
             It manually hides the bottom part of the screen, because
             we are overriding the normal reconstruction of the view
           */
            bottomContainer.setVisibility(View.GONE);
            //It hides the Action bar to give that FullScreen look
            this.getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
                    WindowManager.LayoutParams.FLAG_FULLSCREEN);
            getSupportActionBar().hide();
        } else if (newConfig.orientation == Configuration.ORIENTATION_PORTRAIT) {
            //It overrides the Control Overlay with the Inline controls
            setInlineUI();
            /*
             It manually shows the bottom part of the screen, because
             we are overriding the normal reconstruction of the view
            */
            bottomContainer.setVisibility(View.VISIBLE);
            //It shows the Action bar to return to the normal Activity with Action bar ar the top
            this.getWindow().clearFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);
            getSupportActionBar().show();
        }
    }

    private VideoPlayerContainer.VideoPlayerContainerCallback getVideoPlayerContainerCallback() {
        return new VideoPlayerContainer.VideoPlayerContainerCallback() {
            @Override
            public void onResourceReady(MediaResource mediaResource) {
                videoPlayerView.play(mediaResource);
            }

            @Override
            public void onVideoPlayerCreated() {
                videoPlayerView = videoPlayerContainer.getVideoPlayer();
                //videoPlayerView.setThumbnails("https://mdtp-a.akamaihd.net/preview/preview.vtt", "https://mdtp-a.akamaihd.net/preview/");
                videoPlayerView.setProgressBarControl(progressBar);
                videoPlayerView.setFullScreen(true);
                videoPlayerView.setFullScreenMode(VideoPlayerView.FULLSCREEN_MODE_KEEP_ASPECT_RATIO);
                videoPlayerView.setLogEnabled(true);
                videoPlayerView.enableDVRfeatures(true);  // true activates LIVE tag as button for seekToLive
                //videoPlayerView.shouldAutoPlay(false); // indicates if playback should start automatically
                //videoPlayerView.setRepeatMode(Player.REPEAT_MODE_OFF); // indicates the repeat Mode

                /*
                The controls overlay needs the references of the VideoPlayerContainer to make the
                 connection between the UI components and the player behavior. But, even though the
                 VideoPlayerContainer reference is available since the onCreate, it needs to be assigned her
                 because underneath the controls overlay also needs the videoPlayerView reference, which is
                 obtained through the VideoPlayerView object, and  at this point you know
                 for sure that object is available and ready to use.
                */
                mPlayerControlsOverlay.setVideoPlayerContainer(videoPlayerContainer);
            }

            @Override
            public void onResourceError(ErrorType errorType, Exception exception) {
            }

        };
    }

    private IMediaPlayerControllerListener getMediaPlayerControllerListener() {
        return new IMediaPlayerControllerListener() {
            @Override
            public void onButtonClicked(int buttonId) {

                switch (buttonId) {
                    case IMediaPlayerControllerListener.MEDIAPLAYERCONTROLLER_BUTTON_PLAY:
                        LogManager.log(LOG_TAG, "BUTTON_PLAY");
                        break;
                    case IMediaPlayerControllerListener.MEDIAPLAYERCONTROLLER_BUTTON_PAUSE:
                        LogManager.log(LOG_TAG, "BUTTON_PAUSE");
                        break;
                    case IMediaPlayerControllerListener.MEDIAPLAYERCONTROLLER_BUTTON_SEEK:
                        LogManager.log(LOG_TAG, "BUTTON_SEEK");
                        break;
                    case IMediaPlayerControllerListener.MEDIAPLAYERCONTROLLER_BUTTON_FULLSCREEN:
                        LogManager.log(LOG_TAG, "BUTTON_FULLSCREEN");
                        validateFullScreen();
                        break;
                    case IMediaPlayerControllerListener.MEDIAPLAYERCONTROLLER_CAPTIONS_ON:
                        LogManager.log(LOG_TAG, "CAPTIONS_ON");
                        break;
                    case IMediaPlayerControllerListener.MEDIAPLAYERCONTROLLER_CAPTIONS_OFF:
                        LogManager.log(LOG_TAG, "CAPTIONS_OFF");
                        break;
                    default:
                        break;
                }
            }
        };
    }

    private void validateFullScreen() {
        /*
        The fullscreen approach in this case is based on the current orientation
        so the App just forces a specific one and the config changes method handles the
        action accordingly
        */
        if (mPlayerControlsOverlay.getCurrentOverlayState().isFullScreen()) {
            setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT);
        } else {
            setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE);
        }
    }

    /*
     * The PlayerControlsOverlay is a customized FrameLayout, that contains the required methods to
     * control the basic interaction between the user and a video player, play, pause, replay, scrubbing
     * DVR, etc.
     *
     * The whole point of having a generic UI is that the module is able to handle that basic interaction,
     * regardless of how the UI is displayed in the screen, so you can create yoy own UI as you do normally
     * with any activity or fragment in Android.
     *
     * In order to do that, the module needs as a requirement, the id of the xml that contains the definition of your
     * UI, this is a normal xml layout file, in here, you can create and place any button or UI widget
     * as you prefer.
    * */
    private void setFullScreenUI() {
        mPlayerControlsOverlay.overrideControlsLayout(R.layout.androidsdk_mediaplayercontroller_fullscreen);
        setupFullScreenPlayerControls();
    }

    private void setInlineUI() {
        mPlayerControlsOverlay.overrideControlsLayout(R.layout.androidsdk_mediaplayercontroller);
        setupInlinePlayerControls();
    }

    /*
    * Besides receiving the xml layout of your choosing, you need to let the Controls Overlay know
    * what functionality is going to be handled internally, that's why there is a set of methods available
    * for that.
    *
    * This is important because not only you have the freedom to create your own xml for the controls
    * but you are not restricted to names or specific conditions, you can create the component as you
    * wish, and then just pass the id reference of the specific component, and then the module
    * handles its functionality internally, so you don't have to
    * worry about handling the resources or icons for each button, call the correct player methods or
    * make sure the data displayed is correct, the module those that for you.
    * */
    private void setupFullScreenPlayerControls() {
        //mPlayerControlsOverlay.manageThumbnails(R.id.preview_container);
        mPlayerControlsOverlay.managePlayPause(R.id.androidsdk_playPauseCtrl,
                R.mipmap.amp_play,
                R.mipmap.amp_pause);
        mPlayerControlsOverlay.manageCurrentPosition(R.id.androidsdk_seekbarTextCtrl);
        mPlayerControlsOverlay.manageTimeRemaining(R.id.video_duration);
        mPlayerControlsOverlay.manageScrubbing(R.id.androidsdk_seekbarCtrl,
                R.id.androidsdk_seekToLiveAction);
        mPlayerControlsOverlay.manageFullScreen(R.id.androidsdk_fullscreenCtrl,
                R.mipmap.amp_non_fullscreen_btn,
                R.mipmap.amp_fullscreen_btn);
    }

    private void setupInlinePlayerControls() {
        //mPlayerControlsOverlay.manageThumbnails(R.id.preview_container);
        mPlayerControlsOverlay.managePlayPause(R.id.androidsdk_playPauseCtrl,
                R.mipmap.play,
                R.mipmap.pause);
        mPlayerControlsOverlay.manageCurrentPosition(R.id.androidsdk_seekbarTextCtrl);
        mPlayerControlsOverlay.manageTimeRemaining(R.id.video_duration);
        mPlayerControlsOverlay.manageScrubbing(R.id.androidsdk_seekbarCtrl,
                R.id.androidsdk_seekToLiveAction);

        mPlayerControlsOverlay.manageQualityLevels(R.id.mute_button);

        mPlayerControlsOverlay.manageFullScreen(R.id.androidsdk_fullscreenCtrl,
                R.mipmap.amp_non_fullscreen_btn,
                R.mipmap.amp_fullscreen_btn);

    }

}
