/*
 * Copyright 2019 Google LLC. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.cast.sample.ui;

import android.content.Intent;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.Nullable;
import androidx.fragment.app.Fragment;
import androidx.recyclerview.widget.ItemTouchHelper;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.akamai.amp.cast.AmpCAF;
import com.akamai.amp.cast.AmpCastManager;
import com.akamai.amp.cast.queue.AmpQueueManager;
import com.cast.sample.R;
import com.cast.sample.expandedcontrols.ExpandedControlsActivity;
import com.google.android.gms.cast.MediaQueueItem;

/**
 * A fragment to show the list of queue items.
 */
public class QueueListViewFragment extends Fragment
        implements QueueListAdapter.OnStartDragListener {

    private static final String TAG = "QueueListViewFragment";
    private ItemTouchHelper mItemTouchHelper;

    private AmpCastManager ampCastManager;

    public QueueListViewFragment() {
        super();
    }

    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container,
            @Nullable Bundle savedInstanceState) {
        return inflater.inflate(R.layout.fragment_recycler_list_view, container, false);
    }

    @Override
    public void onStartDrag(RecyclerView.ViewHolder viewHolder) {
        mItemTouchHelper.startDrag(viewHolder);
    }

    @Override
    public void onViewCreated(View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        RecyclerView recyclerView = getView().findViewById(R.id.recycler_view);

        ampCastManager = new AmpCAF.Builder(getContext())
                .withQueue(true)
                .build();

        QueueListAdapter adapter = new QueueListAdapter(getActivity(), this);
        recyclerView.setHasFixedSize(true);
        recyclerView.setAdapter(adapter);
        recyclerView.setLayoutManager(new LinearLayoutManager(getActivity()));

        ItemTouchHelper.Callback callback = new QueueItemTouchHelperCallback(adapter);
        mItemTouchHelper = new ItemTouchHelper(callback);
        mItemTouchHelper.attachToRecyclerView(recyclerView);

        adapter.setEventListener((View v) -> {
                switch (v.getId()) {
                    case R.id.container:
                        Log.d(TAG, "onItemViewClicked() container "
                                + v.getTag(R.string.queue_tag_item));
                        onContainerClicked(v);
                        break;
                    case R.id.play_pause:
                        Log.d(TAG, "onItemViewClicked() play-pause "
                                + v.getTag(R.string.queue_tag_item));
                        getQueue().toggleRemotePlay();
                        break;
                    case R.id.play_upcoming:
                        getQueue().jumpToQueueItem((MediaQueueItem) v.getTag(R.string.queue_tag_item));
                        break;
                    case R.id.stop_upcoming:
                        getQueue().truncateFromQueueItem((MediaQueueItem) v.getTag(R.string.queue_tag_item));
                        break;
                }
            });
    }

    private AmpQueueManager getQueue(){
        return ampCastManager.getQueueManager();
    }


    private void onContainerClicked(View view) {
        MediaQueueItem item = (MediaQueueItem) view.getTag(R.string.queue_tag_item);

        if (getQueue().isQueueDetached()) {
            Log.d(TAG, "Is detached: itemId = " + item.getItemId());
            getQueue().refreshQueueOnCurrentPosition(item);
          } else {
            int currentItemId = getQueue().getCurrentItemId();
            if (currentItemId == item.getItemId()) {
                // We selected the one that is currently playing so we take the user to the
                // full screen controller
                if (ampCastManager.getCastSession() != null) {
                    Intent intent = new Intent(getActivity(), ExpandedControlsActivity.class);
                    startActivity(intent);
                }
            } else {
                // a different item in the queue was selected so we jump there
                getQueue().jumpToQueueItem(item);
            }
        }
    }

    @Override
    public void onActivityCreated(@Nullable Bundle savedInstanceState) {
        super.onActivityCreated(savedInstanceState);
        setRetainInstance(true);
    }
}
