/*
 * Decompiled with CFR 0.152.
 */
package com.akamai.amp.license.decryption;

import com.akamai.amp.license.decryption.AES256v2Ciphertext;
import com.akamai.amp.license.decryption.AES256v3Ciphertext;
import com.akamai.amp.license.decryption.CryptorException;
import com.akamai.amp.license.decryption.InvalidDataException;
import com.akamai.amp.license.decryption.InvalidHMACException;
import com.akamai.amp.license.decryption.JNCryptor;
import com.akamai.amp.license.decryption.PasswordKey;
import com.akamai.amp.license.decryption.Validate;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.SecureRandom;
import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.SecretKeySpec;

public class AES256JNCryptor
implements JNCryptor {
    static final String AES_CIPHER_ALGORITHM = "AES/CBC/PKCS5Padding";
    static final String HMAC_ALGORITHM = "HmacSHA256";
    static final String AES_NAME = "AES";
    static final String KEY_DERIVATION_ALGORITHM = "PBKDF2WithHmacSHA1";
    static final int PBKDF_DEFAULT_ITERATIONS = 10000;
    static final int VERSION = 3;
    static final int AES_256_KEY_SIZE = 32;
    static final int AES_BLOCK_SIZE = 16;
    static final int SALT_LENGTH = 8;
    private static final SecureRandom SECURE_RANDOM = new SecureRandom();
    private int iterations = 10000;

    public AES256JNCryptor() {
    }

    public AES256JNCryptor(int iterations) {
        Validate.isTrue(iterations > 0, "Iteration value must be positive.", new Object[0]);
        this.iterations = iterations;
    }

    @Override
    public SecretKey keyForPassword(char[] password, byte[] salt) throws CryptorException {
        Validate.notNull(salt, "Salt value cannot be null.", new Object[0]);
        Validate.isTrue(salt.length == 8, "Salt value must be %d bytes.", 8);
        Validate.notNull(password, "Password cannot be null.", new Object[0]);
        Validate.isTrue(password.length > 0, "Password cannot be empty.", new Object[0]);
        try {
            SecretKeyFactory factory = SecretKeyFactory.getInstance(KEY_DERIVATION_ALGORITHM);
            SecretKey tmp = factory.generateSecret(new PBEKeySpec(password, salt, this.getPBKDFIterations(), 256));
            return new SecretKeySpec(tmp.getEncoded(), AES_NAME);
        }
        catch (GeneralSecurityException e) {
            throw new CryptorException(String.format("Failed to generate key from password using %s.", KEY_DERIVATION_ALGORITHM), e);
        }
    }

    @Override
    public synchronized int getPBKDFIterations() {
        return this.iterations;
    }

    @Override
    public synchronized void setPBKDFIterations(int iterations) {
        Validate.isTrue(iterations > 0, "Number of iterations must be greater than zero.", new Object[0]);
        this.iterations = iterations;
    }

    private byte[] decryptV2Data(AES256v2Ciphertext aesCiphertext, SecretKey decryptionKey, SecretKey hmacKey) throws CryptorException {
        try {
            Mac mac = Mac.getInstance(HMAC_ALGORITHM);
            mac.init(hmacKey);
            byte[] hmacValue = mac.doFinal(aesCiphertext.getDataToHMAC());
            if (!AES256JNCryptor.arraysEqual(hmacValue, aesCiphertext.getHmac())) {
                throw new InvalidHMACException("Incorrect HMAC value.");
            }
            Cipher cipher = Cipher.getInstance(AES_CIPHER_ALGORITHM);
            cipher.init(2, (Key)decryptionKey, new IvParameterSpec(aesCiphertext.getIv()));
            return cipher.doFinal(aesCiphertext.getCiphertext());
        }
        catch (InvalidKeyException e) {
            throw new CryptorException("Caught InvalidKeyException. Do you have unlimited strength jurisdiction files installed?", e);
        }
        catch (GeneralSecurityException e) {
            throw new CryptorException("Failed to decrypt message.", e);
        }
    }

    private byte[] decryptV3Data(AES256v3Ciphertext aesCiphertext, SecretKey decryptionKey, SecretKey hmacKey) throws CryptorException {
        try {
            Mac mac = Mac.getInstance(HMAC_ALGORITHM);
            mac.init(hmacKey);
            byte[] hmacValue = mac.doFinal(aesCiphertext.getDataToHMAC());
            if (!AES256JNCryptor.arraysEqual(hmacValue, aesCiphertext.getHmac())) {
                throw new InvalidHMACException("Incorrect HMAC value.");
            }
            Cipher cipher = Cipher.getInstance(AES_CIPHER_ALGORITHM);
            cipher.init(2, (Key)decryptionKey, new IvParameterSpec(aesCiphertext.getIv()));
            return cipher.doFinal(aesCiphertext.getCiphertext());
        }
        catch (InvalidKeyException e) {
            throw new CryptorException("Caught InvalidKeyException. Do you have unlimited strength jurisdiction files installed?", e);
        }
        catch (GeneralSecurityException e) {
            throw new CryptorException("Failed to decrypt message.", e);
        }
    }

    @Override
    public byte[] decryptData(byte[] ciphertext, char[] password) throws CryptorException {
        Validate.notNull(ciphertext, "Ciphertext cannot be null.", new Object[0]);
        Validate.notNull(password, "Password cannot be null.", new Object[0]);
        Validate.isTrue(password.length > 0, "Password cannot be empty.", new Object[0]);
        int version = AES256JNCryptor.readVersionNumber(ciphertext);
        switch (version) {
            case 2: {
                return this.decryptV2Data(ciphertext, password);
            }
            case 3: {
                return this.decryptV3Data(ciphertext, password);
            }
        }
        throw new CryptorException(String.format("Unrecognised version number: %d.", version));
    }

    private byte[] decryptV2Data(byte[] ciphertext, char[] password) throws CryptorException {
        try {
            AES256v2Ciphertext aesCiphertext = new AES256v2Ciphertext(ciphertext);
            if (!aesCiphertext.isPasswordBased()) {
                throw new IllegalArgumentException("Ciphertext was not encrypted with a password.");
            }
            SecretKey decryptionKey = this.keyForPassword(password, aesCiphertext.getEncryptionSalt());
            SecretKey hmacKey = this.keyForPassword(password, aesCiphertext.getHmacSalt());
            return this.decryptV2Data(aesCiphertext, decryptionKey, hmacKey);
        }
        catch (InvalidDataException e) {
            throw new CryptorException("Unable to parse ciphertext.", e);
        }
    }

    private byte[] decryptV3Data(byte[] ciphertext, char[] password) throws CryptorException {
        try {
            AES256v3Ciphertext aesCiphertext = new AES256v3Ciphertext(ciphertext);
            if (!aesCiphertext.isPasswordBased()) {
                throw new IllegalArgumentException("Ciphertext was not encrypted with a password.");
            }
            SecretKey decryptionKey = this.keyForPassword(password, aesCiphertext.getEncryptionSalt());
            SecretKey hmacKey = this.keyForPassword(password, aesCiphertext.getHmacSalt());
            return this.decryptV3Data(aesCiphertext, decryptionKey, hmacKey);
        }
        catch (InvalidDataException e) {
            throw new CryptorException("Unable to parse ciphertext.", e);
        }
    }

    @Override
    public byte[] encryptData(byte[] plaintext, char[] password, byte[] encryptionSalt, byte[] hmacSalt, byte[] iv) throws CryptorException {
        Validate.notNull(plaintext, "Plaintext cannot be null.", new Object[0]);
        Validate.notNull(password, "Password cannot be null.", new Object[0]);
        Validate.isTrue(password.length > 0, "Password cannot be empty.", new Object[0]);
        Validate.isCorrectLength(encryptionSalt, 8, "Encryption salt");
        Validate.isCorrectLength(hmacSalt, 8, "HMAC salt");
        Validate.isCorrectLength(iv, 16, "IV");
        SecretKey encryptionKey = this.keyForPassword(password, encryptionSalt);
        SecretKey hmacKey = this.keyForPassword(password, hmacSalt);
        return this.encryptData(plaintext, new PasswordKey(encryptionKey, encryptionSalt), new PasswordKey(hmacKey, hmacSalt), iv);
    }

    @Override
    public byte[] encryptData(byte[] plaintext, char[] password) throws CryptorException {
        Validate.notNull(plaintext, "Plaintext cannot be null.", new Object[0]);
        Validate.notNull(password, "Password cannot be null.", new Object[0]);
        Validate.isTrue(password.length > 0, "Password cannot be empty.", new Object[0]);
        byte[] encryptionSalt = AES256JNCryptor.getSecureRandomData(8);
        byte[] hmacSalt = AES256JNCryptor.getSecureRandomData(8);
        byte[] iv = AES256JNCryptor.getSecureRandomData(16);
        return this.encryptData(plaintext, password, encryptionSalt, hmacSalt, iv);
    }

    static byte[] getSecureRandomData(int length) {
        byte[] result = new byte[length];
        SECURE_RANDOM.nextBytes(result);
        return result;
    }

    @Override
    public int getVersionNumber() {
        return 3;
    }

    @Override
    public byte[] decryptData(byte[] ciphertext, SecretKey decryptionKey, SecretKey hmacKey) throws CryptorException, InvalidHMACException {
        Validate.notNull(ciphertext, "Ciphertext cannot be null.", new Object[0]);
        Validate.notNull(decryptionKey, "Decryption key cannot be null.", new Object[0]);
        Validate.notNull(hmacKey, "HMAC key cannot be null.", new Object[0]);
        try {
            int version = AES256JNCryptor.readVersionNumber(ciphertext);
            switch (version) {
                case 2: {
                    return this.decryptV2Data(new AES256v2Ciphertext(ciphertext), decryptionKey, hmacKey);
                }
                case 3: {
                    return this.decryptV3Data(new AES256v3Ciphertext(ciphertext), decryptionKey, hmacKey);
                }
            }
            throw new CryptorException(String.format("Unrecognised version number: %d.", version));
        }
        catch (InvalidDataException e) {
            throw new CryptorException("Unable to parse ciphertext.", e);
        }
    }

    @Override
    public byte[] encryptData(byte[] plaintext, SecretKey encryptionKey, SecretKey hmacKey) throws CryptorException {
        Validate.notNull(plaintext, "Plaintext cannot be null.", new Object[0]);
        Validate.notNull(encryptionKey, "Encryption key cannot be null.", new Object[0]);
        Validate.notNull(hmacKey, "HMAC key cannot be null.", new Object[0]);
        byte[] iv = AES256JNCryptor.getSecureRandomData(16);
        try {
            Cipher cipher = Cipher.getInstance(AES_CIPHER_ALGORITHM);
            cipher.init(1, (Key)encryptionKey, new IvParameterSpec(iv));
            byte[] ciphertext = cipher.doFinal(plaintext);
            AES256v3Ciphertext output = new AES256v3Ciphertext(iv, ciphertext);
            Mac mac = Mac.getInstance(HMAC_ALGORITHM);
            mac.init(hmacKey);
            byte[] hmac = mac.doFinal(output.getDataToHMAC());
            output.setHmac(hmac);
            return output.getRawData();
        }
        catch (GeneralSecurityException e) {
            throw new CryptorException("Failed to generate ciphertext.", e);
        }
    }

    private static int readVersionNumber(byte[] data) {
        Validate.isTrue(data.length > 0, "Data must be at least one byte long to read version number.", new Object[0]);
        return data[0];
    }

    static boolean arraysEqual(byte[] array1, byte[] array2) {
        if (array1.length != array2.length) {
            return false;
        }
        boolean isEqual = true;
        for (int i = 0; i < array1.length; ++i) {
            if (array1[i] == array2[i]) continue;
            isEqual = false;
        }
        return isEqual;
    }

    @Override
    public PasswordKey getPasswordKey(char[] password) throws CryptorException {
        Validate.notNull(password, "Password cannot be null.", new Object[0]);
        Validate.isTrue(password.length > 0, "Password cannot be empty.", new Object[0]);
        byte[] salt = new byte[8];
        SECURE_RANDOM.nextBytes(salt);
        SecretKey secretKey = this.keyForPassword(password, salt);
        return new PasswordKey(secretKey, salt);
    }

    byte[] encryptData(byte[] plaintext, PasswordKey encryptionKey, PasswordKey hmacKey, byte[] iv) throws CryptorException {
        try {
            Cipher cipher = Cipher.getInstance(AES_CIPHER_ALGORITHM);
            cipher.init(1, (Key)encryptionKey.getKey(), new IvParameterSpec(iv));
            byte[] ciphertext = cipher.doFinal(plaintext);
            AES256v3Ciphertext output = new AES256v3Ciphertext(encryptionKey.getSalt(), hmacKey.getSalt(), iv, ciphertext);
            Mac mac = Mac.getInstance(HMAC_ALGORITHM);
            mac.init(hmacKey.getKey());
            byte[] hmac = mac.doFinal(output.getDataToHMAC());
            output.setHmac(hmac);
            return output.getRawData();
        }
        catch (InvalidKeyException e) {
            throw new CryptorException("Caught InvalidKeyException. Do you have unlimited strength jurisdiction files installed?", e);
        }
        catch (GeneralSecurityException e) {
            throw new CryptorException("Failed to generate ciphertext.", e);
        }
    }

    @Override
    public byte[] encryptData(byte[] plaintext, PasswordKey encryptionKey, PasswordKey hmacKey) throws CryptorException {
        byte[] iv = new byte[16];
        SECURE_RANDOM.nextBytes(iv);
        return this.encryptData(plaintext, encryptionKey, hmacKey, iv);
    }
}

