/*
 * Copyright (C) 2016 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.akamai.downloader.sample;

import android.app.Fragment;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.AssetManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.Uri;
import android.os.AsyncTask;
import android.os.Bundle;
import android.util.JsonReader;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.widget.BaseExpandableListAdapter;
import android.widget.Button;
import android.widget.ExpandableListView;
import android.widget.ExpandableListView.OnChildClickListener;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.fragment.app.FragmentManager;

import com.akamai.amp.downloader.AmpDownloadService;
import com.akamai.amp.downloader.idea.AmpDownloadManager;
import com.akamai.amp.downloader.listener.AmpDownloaderListener;
import com.akamai.amp.downloader.listener.OfflineLicenseDownloadListener;
import com.akamai.amp.downloader.listener.PrepareDownloadListener;
import com.akamai.amp.exoplayer2.C;
import com.akamai.amp.exoplayer2.MediaItem;
import com.akamai.amp.exoplayer2.MediaMetadata;
import com.akamai.amp.exoplayer2.ParserException;
import com.akamai.amp.exoplayer2.offline.Download;
import com.akamai.amp.exoplayer2.offline.DownloadHelper;
import com.akamai.amp.exoplayer2.offline.DownloadService;
import com.akamai.amp.exoplayer2.trackselection.DefaultTrackSelector;
import com.akamai.amp.exoplayer2.trackselection.MappingTrackSelector;
import com.akamai.amp.exoplayer2.upstream.DataSource;
import com.akamai.amp.exoplayer2.upstream.DataSourceInputStream;
import com.akamai.amp.exoplayer2.upstream.DataSpec;
import com.akamai.amp.exoplayer2.upstream.DefaultDataSource;
import com.akamai.amp.exoplayer2.util.Assertions;
import com.akamai.amp.exoplayer2.util.Log;
import com.akamai.amp.exoplayer2.util.Util;
import com.akamai.amp.license.manager.AMPLicenseManager;
import com.akamai.amp.media.elements.MediaResource;
import com.akamai.amp.media.exowrapper2.drm.DrmScheme;
import com.akamai.amp.utils.LogManager;
import com.akamai.amp.utils.MediaResourceUtils;
import com.google.common.collect.ImmutableList;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import static com.akamai.amp.exoplayer2.util.Assertions.checkArgument;
import static com.akamai.amp.exoplayer2.util.Assertions.checkNotNull;
import static com.akamai.amp.exoplayer2.util.Assertions.checkState;

/** An activity for selecting from a list of media samples. */
public class SampleChooserActivity extends AppCompatActivity
        implements AmpDownloaderListener, OnChildClickListener {

  private static final String TAG = "SampleChooserActivity";

  private String[] uris;
  private SampleAdapter sampleAdapter;
  private AmpDownloadManager ampDownloadManager;
  private FrameLayout loadingFragment;
  @Nullable
  private StartDownloadDialogHelper startDownloadDialogHelper;

  private boolean removeProgressAfterCancel = true;

  @Override
  public void onCreate(Bundle savedInstanceState) {
    super.onCreate(savedInstanceState);
    setContentView(R.layout.sample_chooser_activity);
    AMPLicenseManager.setApiKey("demo");
    sampleAdapter = new SampleAdapter();
    ExpandableListView sampleListView = findViewById(R.id.sample_list);
    sampleListView.setAdapter(sampleAdapter);
    loadingFragment = findViewById(R.id.loading_container);
    sampleListView.setOnChildClickListener(this);

    Intent intent = getIntent();
    String dataUri = intent.getDataString();
    if (dataUri != null) {
      uris = new String[] {dataUri};
    } else {
      ArrayList<String> uriList = new ArrayList<>();
      AssetManager assetManager = getAssets();
      try {
        for (String asset : assetManager.list("")) {
          if (asset.endsWith(".exolist.json")) {
            uriList.add("asset:///" + asset);
          }
        }
      } catch (IOException e) {
        Toast.makeText(getApplicationContext(), R.string.sample_list_load_error, Toast.LENGTH_LONG)
                .show();
      }
      uris = new String[uriList.size()];
      uriList.toArray(uris);
      Arrays.sort(uris);
    }

    loadSample();

    ampDownloadManager = SampleApplication.ampDownloadManager;
    // Start the download service if it should be running but it's not currently.
    // Starting the service in the foreground causes notification flicker if there is no scheduled
    // action. Starting it in the background throws an exception if the app is in the background too
    // (e.g. if device screen is locked).
    try {
      DownloadService.start(this, AmpDownloadService.class);
    } catch (IllegalStateException e) {
      DownloadService.startForeground(this, AmpDownloadService.class);
    }
    setupLoading();
    loadingFragment.setVisibility(View.INVISIBLE);
  }

  private void setupLoading() {
    Bundle b = new Bundle();
    b.putCharSequence(LoadingFragment.DESCRIPTION_KEY, "Loading Tracks");

    Fragment f = new LoadingFragment();
    f.setArguments(b);

    getFragmentManager()
            .beginTransaction()
            .replace(R.id.loading_container, f)
            .commit();
  }

  @Override
  public boolean onCreateOptionsMenu(Menu menu) {
    MenuInflater inflater = getMenuInflater();
    inflater.inflate(R.menu.sample_chooser_menu, menu);
    return true;
  }

  @Override
  public boolean onOptionsItemSelected(MenuItem item) {
    if (item.getItemId() == R.id.cancel_all){
      ampDownloadManager.cancelAllDownloads(removeProgressAfterCancel);
    } else if (item.getItemId() == R.id.remove_all){
      ampDownloadManager.removeAllDownloads();
    } else if (item.getItemId() == R.id.pause_all){
      ampDownloadManager.pauseAllDownloads();
    } else if (item.getItemId() == R.id.resume_all){
      ampDownloadManager.resumeAllDownloads();
    }
    return true;
  }

  @Override
  public void onStart() {
    super.onStart();
    ampDownloadManager.addListener(this);
    sampleAdapter.notifyDataSetChanged();
  }

  @Override
  public void onStop() {
    ampDownloadManager.removeListener(this);
    super.onStop();
  }

  @Override
  public void onDownloadsChanged(Download asset) {
    sampleAdapter.notifyDataSetChanged();
    LogManager.log(TAG, "State= " + asset.state);
    if(asset.state == Download.STATE_FAILED){
      Toast.makeText(getApplicationContext(), R.string.exo_download_failed, Toast.LENGTH_SHORT)
              .show();
    }
  }

  @Override
  public void onRequestPermissionsResult(
          int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
    if (grantResults.length == 0) {
      // Empty results are triggered if a permission is requested while another request was already
      // pending and can be safely ignored in this case.
      return;
    }
    if (grantResults[0] == PackageManager.PERMISSION_GRANTED) {
      loadSample();
    } else {
      Toast.makeText(getApplicationContext(), R.string.sample_list_load_error, Toast.LENGTH_LONG)
              .show();
      finish();
    }
  }

  private void loadSample() {
    Assertions.checkNotNull(uris);

    for (int i = 0; i < uris.length; i++) {
      Uri uri = Uri.parse(uris[i]);
      if (Util.maybeRequestReadExternalStoragePermission(this, uri)) {
        return;
      }
    }

    SampleListLoader loaderTask = new SampleListLoader();
    loaderTask.execute(uris);
  }

  private void onPlaylistGroups(final List<PlaylistGroup> groups, boolean sawError) {
    if (sawError) {
      Toast.makeText(getApplicationContext(), R.string.sample_list_load_error, Toast.LENGTH_LONG)
              .show();
    }
    sampleAdapter.setPlaylistGroups(groups);
  }

  @Override
  public boolean onChildClick(
          ExpandableListView parent, View view, int groupPosition, int childPosition, long id) {

    PlaylistHolder playlistHolder = (PlaylistHolder) view.getTag();
    Intent intent = new Intent(this, PlayerActivity.class);
    SampleIntentUtil.addToIntent(playlistHolder.mediaItems, intent);
    startActivity(intent);
    return true;
  }

  private boolean isNetworkAvailable() {
    ConnectivityManager connectivityManager = (ConnectivityManager) getSystemService(Context.CONNECTIVITY_SERVICE);
    NetworkInfo activeNetworkInfo = connectivityManager.getActiveNetworkInfo();
    return activeNetworkInfo != null;
  }


  private void onSampleDownloadButtonClicked(PlaylistHolder playlistHolder, boolean downloadLicenseOnly) {

    int downloadUnsupportedStringId = getDownloadUnsupportedStringId(playlistHolder);
    if (downloadUnsupportedStringId != 0) {
      Toast.makeText(getApplicationContext(), downloadUnsupportedStringId, Toast.LENGTH_SHORT)
              .show();
    } else {
      loadingFragment.setVisibility(View.VISIBLE);
      MediaItem mediaItem = playlistHolder.mediaItems.get(0);
      MediaResource mediaResourceSelected;

      if(mediaItem.playbackProperties.drmConfiguration != null){
        DrmScheme drmScheme;
        UUID uuid = mediaItem.playbackProperties.drmConfiguration.uuid;
        if(DrmScheme.WIDEVINE.getUuid().equals(uuid)){
          drmScheme = DrmScheme.WIDEVINE;
        } else if(DrmScheme.CLEARKEY.getUuid().equals(uuid)){
          drmScheme = DrmScheme.CLEARKEY;
        } else {
          drmScheme = DrmScheme.PLAYREADY;
        }
        mediaResourceSelected = MediaResourceUtils.createMediaResourceFromURL(mediaItem.playbackProperties.uri.toString(),
                drmScheme,
                mediaItem.playbackProperties.drmConfiguration.licenseUri.toString());
      } else {
        mediaResourceSelected = MediaResourceUtils.createMediaResourceFromURL(mediaItem.playbackProperties.uri.toString());
      }

      mediaResourceSelected.setTitle(mediaItem.mediaMetadata.title);
      mediaResourceSelected.setMimeType(mediaItem.playbackProperties.mimeType);

      if(ampDownloadManager.isVideoDownloaded(mediaItem.playbackProperties.uri) && !downloadLicenseOnly){
        ampDownloadManager.deleteDownload(mediaItem.playbackProperties.uri);
        Toast.makeText(getApplicationContext(), R.string.exo_download_removed, Toast.LENGTH_SHORT)
                .show();
        loadingFragment.setVisibility(View.INVISIBLE);
      } else {
        if(!isNetworkAvailable()){
          Toast.makeText(getApplicationContext(), R.string.no_connection_message, Toast.LENGTH_SHORT)
                  .show();
          loadingFragment.setVisibility(View.INVISIBLE);
        } else {
          if(downloadLicenseOnly){
            downloadLicenseOnly(mediaResourceSelected);
          } else {
            downloadVideo(mediaResourceSelected);
          }
        }
      }
    }
  }

  private void downloadVideo(MediaResource mediaResourceSelected){
    if (startDownloadDialogHelper != null) {
      startDownloadDialogHelper.release();
    }
    HashMap<String, String> headers = new  HashMap<>();
    headers.put("headerKey", "heaverValue");
    ampDownloadManager.toggleDownload(this, mediaResourceSelected, new PrepareDownloadListener() {
      @Override
      public void onSingleDownloadOptionAvailable(MediaResource mediaResource) {
        ampDownloadManager.startDownload(mediaResource);
        loadingFragment.setVisibility(View.INVISIBLE);
      }

      @Override
      public void onMultipleDownloadOptionsAvailable(MediaResource mediaResource,
                                                     MappingTrackSelector.MappedTrackInfo mappedTrackInfo,
                                                     DefaultTrackSelector.Parameters trackSelectorParameters) {
        startDownloadDialogHelper =
                new StartDownloadDialogHelper(getSupportFragmentManager(), mediaResource);
        startDownloadDialogHelper.displayDownloadDialog(mappedTrackInfo, trackSelectorParameters);
        loadingFragment.setVisibility(View.INVISIBLE);
      }

      @Override
      public void onFailed(IOException e) {
        onError(SampleChooserActivity.this, e);
        loadingFragment.setVisibility(View.INVISIBLE);
      }
    }, headers);
  }

  private void downloadLicenseOnly(MediaResource mediaResourceSelected){
    HashMap<String, String> headers = new  HashMap<>();
    headers.put("headerKey", "heaverValue");
    ampDownloadManager.updateDownloadDRMConfiguration(this, mediaResourceSelected, new OfflineLicenseDownloadListener() {
      @Override
      public void onLicenseFetch(MediaResource mediaResource, DownloadHelper helper, byte[] keySetId) {
        loadingFragment.setVisibility(View.INVISIBLE);
      }

      @Override
      public void onFailed(IOException e) {
        onError(SampleChooserActivity.this, e);
        loadingFragment.setVisibility(View.INVISIBLE);
      }
    }, headers);
  }

  public void onError(Context context, IOException e) {
    sampleAdapter.notifyDataSetChanged();
    Toast.makeText(context, R.string.download_start_error, Toast.LENGTH_LONG)
            .show();
    android.util.Log.e(TAG, "Failed to start download", e);
  }

  private int getDownloadUnsupportedStringId(PlaylistHolder playlistHolder) {
    if (playlistHolder.mediaItems.size() > 1) {
      return R.string.download_playlist_unsupported;
    }
    MediaItem.PlaybackProperties playbackProperties =
            checkNotNull(playlistHolder.mediaItems.get(0).playbackProperties);
    if (playbackProperties.adTagUri != null) {
      return R.string.download_ads_unsupported;
    }
    String scheme = playbackProperties.uri.getScheme();
    if (!("http".equals(scheme) || "https".equals(scheme))) {
      return R.string.download_scheme_unsupported;
    }
    return 0;
  }

  private final class SampleListLoader extends AsyncTask<String, Void, List<PlaylistGroup>> {

    private boolean sawError;

    @Override
    protected List<PlaylistGroup> doInBackground(String... uris) {
      List<PlaylistGroup> result = new ArrayList<>();
      Context context = getApplicationContext();
      String userAgent = Util.getUserAgent(context, "AMPDownloaderDemo");
      DataSource dataSource =
              new DefaultDataSource(context, userAgent, /* allowCrossProtocolRedirects= */ false);
      for (String uri : uris) {
        DataSpec dataSpec = new DataSpec(Uri.parse(uri));
        InputStream inputStream = new DataSourceInputStream(dataSource, dataSpec);
        try {
          readPlaylistGroups(new JsonReader(new InputStreamReader(inputStream, "UTF-8")), result);
        } catch (Exception e) {
          Log.e(TAG, "Error loading sample list: " + uri, e);
          sawError = true;
        } finally {
          Util.closeQuietly(dataSource);
        }
      }
      return result;
    }

    @Override
    protected void onPostExecute(List<PlaylistGroup> result) {
      onPlaylistGroups(result, sawError);
    }

    private void readPlaylistGroups(JsonReader reader, List<PlaylistGroup> groups)
            throws IOException {
      reader.beginArray();
      while (reader.hasNext()) {
        readPlaylistGroup(reader, groups);
      }
      reader.endArray();
    }

    private void readPlaylistGroup(JsonReader reader, List<PlaylistGroup> groups)
            throws IOException {
      String groupName = "";
      ArrayList<PlaylistHolder> playlistHolders = new ArrayList<>();

      reader.beginObject();
      while (reader.hasNext()) {
        String name = reader.nextName();
        switch (name) {
          case "name":
            groupName = reader.nextString();
            break;
          case "samples":
            reader.beginArray();
            while (reader.hasNext()) {
              playlistHolders.add(readEntry(reader, false));
            }
            reader.endArray();
            break;
          case "_comment":
            reader.nextString(); // Ignore.
            break;
          default:
            throw new ParserException("Unsupported name: " + name);
        }
      }
      reader.endObject();

      PlaylistGroup group = getGroup(groupName, groups);
      group.playlists.addAll(playlistHolders);
    }

    private PlaylistHolder readEntry(JsonReader reader, boolean insidePlaylist) throws IOException {
      Uri uri = null;
      String extension = null;
      String title = null;
      ArrayList<PlaylistHolder> children = null;
      Uri subtitleUri = null;
      String subtitleMimeType = null;
      String subtitleLanguage = null;

      MediaItem.Builder mediaItem = new MediaItem.Builder();
      reader.beginObject();
      while (reader.hasNext()) {
        String name = reader.nextName();
        switch (name) {
          case "name":
            title = reader.nextString();
            break;
          case "uri":
            uri = Uri.parse(reader.nextString());
            break;
          case "extension":
            extension = reader.nextString();
            break;
          case "clip_start_position_ms":
            mediaItem.setClipStartPositionMs(reader.nextLong());
            break;
          case "clip_end_position_ms":
            mediaItem.setClipEndPositionMs(reader.nextLong());
            break;
          case "ad_tag_uri":
            mediaItem.setAdTagUri(reader.nextString());
            break;
          case "drm_scheme":
            mediaItem.setDrmUuid(Util.getDrmUuid(reader.nextString()));
            break;
          case "drm_license_uri":
          case "drm_license_url": // For backward compatibility only.
            mediaItem.setDrmLicenseUri(reader.nextString());
            break;
          case "drm_key_request_properties":
            Map<String, String> requestHeaders = new HashMap<>();
            reader.beginObject();
            while (reader.hasNext()) {
              requestHeaders.put(reader.nextName(), reader.nextString());
            }
            reader.endObject();
            mediaItem.setDrmLicenseRequestHeaders(requestHeaders);
            break;
          case "drm_session_for_clear_content":
            if (reader.nextBoolean()) {
              mediaItem.setDrmSessionForClearTypes(
                      ImmutableList.of(C.TRACK_TYPE_VIDEO, C.TRACK_TYPE_AUDIO));
            }
            break;
          case "drm_multi_session":
            mediaItem.setDrmMultiSession(reader.nextBoolean());
            break;
          case "drm_force_default_license_uri":
            mediaItem.setDrmForceDefaultLicenseUri(reader.nextBoolean());
            break;
          case "subtitle_uri":
            subtitleUri = Uri.parse(reader.nextString());
            break;
          case "subtitle_mime_type":
            subtitleMimeType = reader.nextString();
            break;
          case "subtitle_language":
            subtitleLanguage = reader.nextString();
            break;
          case "playlist":
            checkState(!insidePlaylist, "Invalid nesting of playlists");
            children = new ArrayList<>();
            reader.beginArray();
            while (reader.hasNext()) {
              children.add(readEntry(reader, /* insidePlaylist= */ true));
            }
            reader.endArray();
            break;
          default:
            throw new ParserException("Unsupported attribute name: " + name);
        }
      }
      reader.endObject();

      if (children != null) {
        List<MediaItem> mediaItems = new ArrayList<>();
        for (int i = 0; i < children.size(); i++) {
          mediaItems.addAll(children.get(i).mediaItems);
        }
        return new PlaylistHolder(title, mediaItems);
      } else {
        @Nullable
        String adaptiveMimeType =
                Util.getAdaptiveMimeTypeForContentType(Util.inferContentType(uri, extension));
        mediaItem
                .setUri(uri)
                .setMediaMetadata(new MediaMetadata.Builder().setTitle(title).build())
                .setMimeType(adaptiveMimeType);
        if (subtitleUri != null) {
          MediaItem.Subtitle subtitle =
                  new MediaItem.Subtitle(
                          subtitleUri,
                          checkNotNull(
                                  subtitleMimeType, "subtitle_mime_type is required if subtitle_uri is set."),
                          subtitleLanguage);
          mediaItem.setSubtitles(Collections.singletonList(subtitle));
        }
        return new PlaylistHolder(title, Collections.singletonList(mediaItem.build()));
      }
    }

    private PlaylistGroup getGroup(String groupName, List<PlaylistGroup> groups) {
      for (int i = 0; i < groups.size(); i++) {
        if (Util.areEqual(groupName, groups.get(i).title)) {
          return groups.get(i);
        }
      }
      PlaylistGroup group = new PlaylistGroup(groupName);
      groups.add(group);
      return group;
    }
  }

  private void cancelSingleDownload(PlaylistHolder playlistHolder){
    MediaItem mediaItem = playlistHolder.mediaItems.get(0);
    ampDownloadManager.cancelDownload(mediaItem.playbackProperties.uri, removeProgressAfterCancel);
    Toast.makeText(getApplicationContext(), R.string.cancel_confirmation, Toast.LENGTH_LONG)
            .show();
  }

  private final OnClickListener downloadOnClickListener = (View view) -> onSampleDownloadButtonClicked((PlaylistHolder) view.getTag(), false);

  private final OnClickListener downloadLicenseOnClickListener = (View view) -> onSampleDownloadButtonClicked((PlaylistHolder) view.getTag(), true);

  private final OnClickListener cancelOnClickListener = (View view) -> cancelSingleDownload((PlaylistHolder) view.getTag());

  private final class SampleAdapter extends BaseExpandableListAdapter {

    private List<PlaylistGroup> playlistGroups;

    public SampleAdapter() {
      playlistGroups = Collections.emptyList();
    }

    public void setPlaylistGroups(List<PlaylistGroup> playlistGroups) {
      this.playlistGroups = playlistGroups;
      notifyDataSetChanged();
    }

    @Override
    public PlaylistHolder getChild(int groupPosition, int childPosition) {
      return getGroup(groupPosition).playlists.get(childPosition);
    }

    @Override
    public long getChildId(int groupPosition, int childPosition) {
      return childPosition;
    }

    @Override
    public View getChildView(int groupPosition, int childPosition, boolean isLastChild,
                             View convertView, ViewGroup parent) {
      View view = convertView;
      if (view == null) {
        view = getLayoutInflater().inflate(R.layout.sample_list_item, parent, false);
        View downloadButton = view.findViewById(R.id.download_button);
        View getLicenseButton = view.findViewById(R.id.download_license_button);
        View cancelButton = view.findViewById(R.id.cancel_button);
        downloadButton.setOnClickListener(downloadOnClickListener);
        getLicenseButton.setOnClickListener(downloadLicenseOnClickListener);
        cancelButton.setOnClickListener(cancelOnClickListener);
        getLicenseButton.setFocusable(false);
        downloadButton.setFocusable(false);
        cancelButton.setFocusable(false);
      }
      initializeChildView(view, getChild(groupPosition, childPosition));
      return view;
    }

    @Override
    public int getChildrenCount(int groupPosition) {
      return getGroup(groupPosition).playlists.size();
    }

    @Override
    public PlaylistGroup getGroup(int groupPosition) {
      return playlistGroups.get(groupPosition);
    }

    @Override
    public long getGroupId(int groupPosition) {
      return groupPosition;
    }

    @Override
    public View getGroupView(
            int groupPosition, boolean isExpanded, View convertView, ViewGroup parent) {
      View view = convertView;
      if (view == null) {
        view =
                getLayoutInflater()
                        .inflate(android.R.layout.simple_expandable_list_item_1, parent, false);
      }
      ((TextView) view).setText(getGroup(groupPosition).title);
      return view;
    }

    @Override
    public int getGroupCount() {
      return playlistGroups.size();
    }

    @Override
    public boolean hasStableIds() {
      return false;
    }

    @Override
    public boolean isChildSelectable(int groupPosition, int childPosition) {
      return true;
    }

    private void initializeChildView(View view, PlaylistHolder playlistHolder) {
      view.setTag(playlistHolder);
      TextView sampleTitle = view.findViewById(R.id.sample_title);
      sampleTitle.setText(playlistHolder.title);

      boolean canDownload = getDownloadUnsupportedStringId(playlistHolder) == 0;
      boolean isDownloaded = canDownload && ampDownloadManager.isVideoDownloaded(playlistHolder.mediaItems.get(0).playbackProperties.uri);
      ImageButton downloadButton = view.findViewById(R.id.download_button);
      downloadButton.setTag(playlistHolder);
      downloadButton.setColorFilter(
              canDownload ? (isDownloaded ? 0xFF42A5F5 : 0xFFBDBDBD) : 0xFF666666);
      downloadButton.setImageResource(
              isDownloaded ? R.drawable.ic_download_done : R.drawable.ic_download);
      Button cancelButton = view.findViewById(R.id.cancel_button);
      cancelButton.setTag(playlistHolder);
      Button getLicenseButton = view.findViewById(R.id.download_license_button);
      getLicenseButton.setTag(playlistHolder);
    }
  }

  private final class StartDownloadDialogHelper
          implements  DialogInterface.OnClickListener,
          DialogInterface.OnDismissListener {

    private final FragmentManager fragmentManager;

    private TrackSelectionDialog trackSelectionDialog;
    private MediaResource mediaResource;

    public StartDownloadDialogHelper(FragmentManager fragmentManager, MediaResource mediaResource) {
      this.fragmentManager = fragmentManager;
      this.mediaResource = mediaResource;
    }

    public void release() {
      if (trackSelectionDialog != null) {
        trackSelectionDialog.dismiss();
      }
    }

    public void displayDownloadDialog(MappingTrackSelector.MappedTrackInfo mappedTrackInfo,
                                      DefaultTrackSelector.Parameters trackSelectorParameters){
      trackSelectionDialog =
              TrackSelectionDialog.createForMappedTrackInfoAndParameters(
                      /* titleId= */ com.akamai.amp.downloader.R.string.exo_download_description,
                      mappedTrackInfo,
                      trackSelectorParameters,
                      /* allowAdaptiveSelections =*/ false,
                      /* allowMultipleOverrides= */ true,
                      /* onClickListener= */ this,
                      /* onDismissListener= */ this);
      trackSelectionDialog.show(fragmentManager, /* tag= */ null);
    }

    // DialogInterface.OnClickListener implementation.

    @Override
    public void onClick(DialogInterface dialog, int which) {
      ampDownloadManager.extractSelectedRenderers(trackSelectionDialog);
      ampDownloadManager.startDownload(mediaResource);
    }

    // DialogInterface.OnDismissListener implementation.

    @Override
    public void onDismiss(DialogInterface dialogInterface) {
      trackSelectionDialog = null;
      ampDownloadManager.cancelToggle();
    }
  }

  private static final class PlaylistHolder {

    public final String title;
    public final List<MediaItem> mediaItems;

    private PlaylistHolder(String title, List<MediaItem> mediaItems) {
      checkArgument(!mediaItems.isEmpty());
      this.title = title;
      this.mediaItems = Collections.unmodifiableList(new ArrayList<>(mediaItems));
    }
  }

  private static final class PlaylistGroup {

    public final String title;
    public final List<PlaylistHolder> playlists;

    public PlaylistGroup(String title) {
      this.title = title;
      this.playlists = new ArrayList<>();
    }
  }
}
