package com.akamai.ampdemo;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.util.Log;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.ProgressBar;

import com.akamai.amp.analytics.comscorestreamsense.AmpComscoreStreamsenseAnalyticsTracker;
import com.akamai.amp.analytics.comscorestreamsense.ComscoreStreamsense;
import com.akamai.amp.analytics.comscorestreamsense.UserConsent;
import com.akamai.amp.config.data.streamsense.StreamsenseData;
import com.akamai.amp.media.VideoPlayerContainer;
import com.akamai.amp.media.VideoPlayerView;
import com.akamai.amp.media.elements.MediaResource;
import com.akamai.amp.media.errors.ErrorType;
import com.akamai.amp.uimobile.generic.media.PlayerControlsOverlay;
import com.akamai.amp.utils.LogManager;
import com.comscore.streaming.ContentMetadata;

import static android.os.AsyncTask.execute;


public class MainActivity extends Activity
        implements VideoPlayerContainer.VideoPlayerContainerCallback {

    private static final String LOG_TAG = "AMPv6";
    private static final String VIDEO_URL = "https://welcomevideo.akamaized.net/delivery/1f/07/1f0756a1-f54b-4e22-adbc-8c68680532ea/tom-intro-for-new-customers-v26b238090-d41d-336d-4347-3dac7fc07f0e.m3u8";

    private VideoPlayerContainer videoPlayerContainer;
    private VideoPlayerView videoPlayerView;
    private ProgressBar progressBar;
    private PlayerControlsOverlay mPlayerControlsOverlay;

    private AmpComscoreStreamsenseAnalyticsTracker streamsenseAnalytics;


    // TODO In order to debug with Charles proxy, and see the outgoing beacons,
    // TODO it might be necessary to add the "myca.pem" certificate in the /res/raw folder
    // Optionally, the <network-security-config> on the network_security_config.xml file should be left empty.


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        progressBar = findViewById(R.id.progressBar);
        mPlayerControlsOverlay = (PlayerControlsOverlay) findViewById(R.id.playerControls);
        setInlineUI();
        videoPlayerContainer = findViewById(R.id.playerViewCtrl);
        videoPlayerContainer.addVideoPlayerContainerCallback(this);
        videoPlayerContainer.setApiKey("demo");
        videoPlayerContainer.prepareResource(VIDEO_URL);

        Button userConsentButton = findViewById(R.id.userConsentButton);
        userConsentButton.setOnClickListener(view -> userConsentPopup());
    }

    private void setInlineUI() {
        mPlayerControlsOverlay.overrideControlsLayout(R.layout.overlay_controls);
        mPlayerControlsOverlay.setShowControlsAtStart(false);
        setupInlinePlayerControls();
    }

    private void setupInlinePlayerControls() {
        mPlayerControlsOverlay.managePlayPause(R.id.androidsdk_playPauseCtrl,
                R.drawable.play,
                R.drawable.pause);
        mPlayerControlsOverlay.manageCurrentPosition(R.id.androidsdk_seekbarTextCtrl);
        mPlayerControlsOverlay.manageTimeRemaining(R.id.video_duration);
        mPlayerControlsOverlay.manageScrubbing(R.id.androidsdk_seekbarCtrl,
                R.id.androidsdk_seekToLiveAction);
    }

    private void userConsentPopup() {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setMessage("Do you provide consent?")
                .setPositiveButton("YES", dialogClickListener)
                .setNegativeButton("NO", dialogClickListener)
                .show();
    }

    private final DialogInterface.OnClickListener dialogClickListener = new DialogInterface.OnClickListener() {
        @Override
        public void onClick(DialogInterface dialog, int which) {
            /*
             * To communicate user consent, a publisher must add the "cs_ucfr" label.
             * AMP has abstracted those values with the UserConsent enum.
             * The app developer doesn't need to call any other methods, besides the ones shown here
             *
             * The underlying values are:
             * "0": User has not given consent or has opted out
             * "1": User has given consent
             * "": Use an empty string value (i.e., blank) to indicate the user has not taken an action
             */

            switch (which) {
                case DialogInterface.BUTTON_POSITIVE:
                    onUserConsentSelected(true);
                    break;

                case DialogInterface.BUTTON_NEGATIVE:
                    onUserConsentSelected(false);
                    break;
            }
        }
    };

    private void onUserConsentSelected(final boolean consent) {
        notifyStreamsenseOfUserConsent(consent);
        execute(() -> saveUserConsentPreference(consent));
    }

    private void notifyStreamsenseOfUserConsent(boolean consent) {
        UserConsent selectedUserConsent = consent ? UserConsent.CONSENT : UserConsent.NO_CONSENT;
        LogManager.log(LOG_TAG, "Selected User Consent: " + selectedUserConsent);

        if (streamsenseAnalytics != null) {
            streamsenseAnalytics.setUserConsent(selectedUserConsent);
        }
    }

    private void saveUserConsentPreference(boolean consent) {
        SharedPreferences sharedPreferences = getSharedPreferences(StreamsenseApplication.PREFERENCES_NAME, Context.MODE_PRIVATE);
        SharedPreferences.Editor editor = sharedPreferences.edit();
        editor.putBoolean(StreamsenseApplication.USER_CONSENT_KEY, consent);
        editor.apply();
    }

    @Override
    public void onResourceReady(MediaResource mediaResource) {
        videoPlayerView = videoPlayerContainer.getVideoPlayer();
        videoPlayerView.setProgressBarControl(progressBar);
        videoPlayerView.setFullScreen(true);
        videoPlayerView.setFullScreenMode(VideoPlayerView.FULLSCREEN_MODE_KEEP_ASPECT_RATIO);
        videoPlayerView.setLogEnabled(true);

        mPlayerControlsOverlay.setVideoPlayerContainer(videoPlayerContainer);
        StreamsenseData streamsenseData = StreamsenseHelper.buildStreamsenseData(videoPlayerView);
        streamsenseAnalytics = ComscoreStreamsense.create(videoPlayerView, streamsenseData);
        ContentMetadata contentMetadata = StreamsenseHelper.buildContentMetadata();
        streamsenseAnalytics.setContentMetadata(contentMetadata);

        videoPlayerView.play(mediaResource);
    }

    @Override
    public void onVideoPlayerCreated() {
        Log.i(LOG_TAG, "onVideoPlayerCreated()");
    }

    @Override
    public void onResourceError(ErrorType errorType, Exception exception) {
        Log.e(LOG_TAG, "onResourceError()");
    }


    //It's mandatory to let the SDK know when the app goes to and comes from the background

    @Override
    protected void onPause() {
        Log.d(LOG_TAG, "onPause");
        if (videoPlayerView != null) videoPlayerView.onPause();
        super.onPause();
    }

    @Override
    protected void onResume() {
        Log.d(LOG_TAG, "onResume");
        if (videoPlayerView != null) videoPlayerView.onResume();
        super.onResume();
    }

    @Override
    protected void onDestroy() {
        Log.d(LOG_TAG, "onDestroy");
        if (videoPlayerView != null) videoPlayerView.onDestroy();
        super.onDestroy();
    }

}
