/*
 * Copyright 2019 Google LLC. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.cast.sample.mediaplayer;

import android.annotation.SuppressLint;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Point;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.text.method.ScrollingMovementMethod;
import android.util.Log;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.view.WindowManager;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.SeekBar;
import android.widget.SeekBar.OnSeekBarChangeListener;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.Toolbar;
import androidx.core.app.ActivityCompat;
import androidx.core.view.ViewCompat;
import androidx.mediarouter.app.MediaRouteButton;

import com.akamai.amp.cast.AmpCAF;
import com.akamai.amp.cast.AmpCastManager;
import com.akamai.amp.cast.queue.RemoteLoadListener;
import com.akamai.amp.cast.utils.AmpCastUtils;
import com.akamai.amp.cast.values.PlaybackLocation;
import com.akamai.amp.cast.values.PlaybackState;
import com.akamai.amp.media.IPlayerEventsListener;
import com.akamai.amp.media.VideoPlayerContainer;
import com.akamai.amp.media.VideoPlayerView;
import com.akamai.amp.media.elements.MediaResource;
import com.akamai.amp.media.errors.ErrorType;
import com.cast.sample.R;
import com.cast.sample.browser.VideoProvider;
import com.cast.sample.expandedcontrols.ExpandedControlsActivity;
import com.cast.sample.settings.CastPreference;
import com.cast.sample.ui.QueueListViewActivity;
import com.cast.sample.utils.Utils;
import com.google.android.gms.cast.MediaInfo;
import com.google.android.gms.cast.MediaMetadata;
import com.google.android.gms.cast.framework.CastSession;
import com.google.android.gms.cast.framework.SessionManagerListener;
import com.google.android.gms.cast.framework.media.MediaUtils;
import java.util.Timer;
import java.util.TimerTask;

/**
 * Activity for the local media player.
 */
public class LocalPlayerActivity extends AppCompatActivity {

    private static final String TAG = "LocalPlayerActivity";
    private TextView mTitleView;
    private TextView mDescriptionView;
    private TextView mStartText;
    private TextView mEndText;
    private SeekBar mSeekbar;
    private ImageView mPlayPause;
    private ProgressBar mLoading;
    private View mControllers;
    private View mContainer;
    private Timer mSeekbarTimer;
    private Timer mControllersTimer;
    private PlaybackState mPlaybackState;
    private final Handler mHandler = new Handler();
    private final float mAspectRatio = 72f / 128;
    private boolean mControllersVisible;
    private int mDuration;
    private TextView mAuthorView;
    private ImageButton mPlayCircle;
    private SessionManagerListener<CastSession> mSessionManagerListener;
    private MenuItem mQueueMenuItem;
    private MediaRouteButton mediaRouteButton;

    private AmpCastManager ampCastManager;
    private VideoPlayerContainer videoPlayerContainer;
    private VideoPlayerView videoPlayerView;
    private int currentPosition;
    private MediaInfo mSelectedMedia;

    private RemoteLoadListener remoteLoadListener = () -> {
            Intent intent = new Intent(LocalPlayerActivity.this, ExpandedControlsActivity.class);
            startActivity(intent);
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.player_activity);
        loadViews();
        setupControlsCallbacks();
        setupCastListener();
        ampCastManager = new AmpCAF.Builder(this)
                .withSessionManagerListener(mSessionManagerListener)
                .withVideoPlayerContainer(videoPlayerContainer)
                .withQueue(true)
                .build();
        AmpCastUtils.initCastButton(this, mediaRouteButton);


        // see what we need to play and where
        Bundle bundle = getIntent().getExtras();
        if (bundle != null) {
            mSelectedMedia = getIntent().getParcelableExtra("media");
            setupActionBar();
            boolean shouldStartPlayback = bundle.getBoolean("shouldStart");
            currentPosition = bundle.getInt("startPosition", 0);
            videoPlayerContainer.loadPoster(MediaUtils.getImageUrl(mSelectedMedia, 0));
            videoPlayerContainer.addVideoPlayerContainerCallback(new VideoPlayerContainer.VideoPlayerContainerCallback() {
                @Override
                public void onVideoPlayerCreated() {
                    videoPlayerView = videoPlayerContainer.getVideoPlayer();
                    videoPlayerView.addOnTouchListener((View v, MotionEvent event) -> {
                        v.performClick();
                        if (!mControllersVisible) {
                            updateControllersVisibility(true);
                        }
                        startControllersTimer();
                        return false;
                    });
                    videoPlayerView.setProgressBarControl(mLoading);
                    videoPlayerView.addEventsListener(new IPlayerEventsListener() {
                        @Override
                        public boolean onPlayerEvent(int i) {
                            if(i == PLAYER_EVENT_TYPE_STARTED){
                                mDuration = videoPlayerView.getTimelineDuration();
                                mEndText.setText(Utils.formatMillis(mDuration));
                                mSeekbar.setMax(mDuration);
                                restartTrickplayTimer();
                            } else if(i == PLAYER_EVENT_TYPE_FINISHED) {
                                stopTrickplayTimer();
                                Log.d(TAG, "setOnCompletionListener()");
                                mPlaybackState = PlaybackState.IDLE;
                                videoPlayerContainer.loadPoster(MediaUtils.getImageUrl(mSelectedMedia, 0));
                                updatePlayButton(mPlaybackState);
                            }
                            return false;
                        }

                        @Override
                        public boolean onPlayerExtendedEvent(int i, int i1, int i2) {
                            return false;
                        }
                    });

                }

                @Override
                public void onResourceReady(MediaResource mediaResource) {
                    videoPlayerView.play(mediaResource, currentPosition);
                }

                @Override
                public void onResourceError(ErrorType errorType, Exception e) {
                    Utils.showErrorDialog(LocalPlayerActivity.this, e.getLocalizedMessage());
                    mPlaybackState = PlaybackState.IDLE;
                    updatePlayButton(mPlaybackState);
                }
            });

            if (shouldStartPlayback) {
                // this will be the case only if we are coming from the
                // CastControllerActivity by disconnecting from a device
                mPlaybackState = PlaybackState.PLAYING;
                updatePlaybackLocation();
                updatePlayButton(mPlaybackState);
                videoPlayerContainer.prepareResource(mSelectedMedia.getContentId());
                startControllersTimer();
            } else {
                // we should load the video but pause it
                // and show the album art.
                updatePlaybackLocation();
                mPlaybackState = PlaybackState.IDLE;
                updatePlayButton(mPlaybackState);
            }
        }
        if (mTitleView != null) {
            updateMetadata(true);
        }
    }

    private void setupCastListener() {
        mSessionManagerListener = new SessionManagerListener<CastSession>() {

            @Override
            public void onSessionEnded(CastSession session, int error) {
                onApplicationDisconnected();
            }

            @Override
            public void onSessionResumed(CastSession session, boolean wasSuspended) {
                onApplicationConnected();
            }

            @Override
            public void onSessionResumeFailed(CastSession session, int error) {
                onApplicationDisconnected();
            }

            @Override
            public void onSessionStarted(CastSession session, String sessionId) {
                onApplicationConnected();
            }

            @Override
            public void onSessionStartFailed(CastSession session, int error) {
                onApplicationDisconnected();
            }

            @Override
            public void onSessionStarting(CastSession session) {
            }

            @Override
            public void onSessionEnding(CastSession session) {
            }

            @Override
            public void onSessionResuming(CastSession session, String sessionId) {
            }

            @Override
            public void onSessionSuspended(CastSession session, int reason) {
            }

            private void onApplicationConnected() {
                if (null != mSelectedMedia) {

                    if (mPlaybackState == PlaybackState.PLAYING) {
                        videoPlayerView.pause();
                        currentPosition = videoPlayerView.getCurrentTimelinePosition();
                        ampCastManager.getQueueManager().loadRemoteMedia(mSelectedMedia, currentPosition *1000, true, remoteLoadListener);
                        return;
                    } else {
                        mPlaybackState = PlaybackState.IDLE;
                        updatePlaybackLocation();
                    }
                }
                updatePlayButton(mPlaybackState);
                invalidateOptionsMenu();
            }

            private void onApplicationDisconnected() {
                updatePlaybackLocation();
                mPlaybackState = PlaybackState.IDLE;
                updatePlayButton(mPlaybackState);
                invalidateOptionsMenu();
            }
        };
    }

    private void updatePlaybackLocation() {
        PlaybackLocation location = ampCastManager.getPlaybackLocation();
        if (location == PlaybackLocation.LOCAL) {
            if (mPlaybackState == PlaybackState.PLAYING
                    || mPlaybackState == PlaybackState.BUFFERING) {
                startControllersTimer();
            } else {
                stopControllersTimer();
            }
        } else {
            stopControllersTimer();
            updateControllersVisibility(false);
        }
    }

    private void play(int position) {
        startControllersTimer();
        switch (ampCastManager.getPlaybackLocation()) {
            case LOCAL:
                videoPlayerView.seek(position);
                break;
            case REMOTE:
                mPlaybackState = PlaybackState.BUFFERING;
                updatePlayButton(mPlaybackState);
                ampCastManager.getCastSession().getRemoteMediaClient().seek(position);
                break;
            default:
                break;
        }
        restartTrickplayTimer();
    }

    private void togglePlayback() {
        stopControllersTimer();
        switch (mPlaybackState) {
            case PAUSED:
                switch (ampCastManager.getPlaybackLocation()) {
                    case LOCAL:
                        videoPlayerView.resume();
                        Log.d(TAG, "Playing locally...");
                        mPlaybackState = PlaybackState.PLAYING;
                        startControllersTimer();
                        restartTrickplayTimer();
                        updatePlaybackLocation();
                        break;
                    case REMOTE:
                        ampCastManager.getQueueManager().loadRemoteMedia(mSelectedMedia, 0, true, remoteLoadListener);
                        finish();
                        break;
                    default:
                        break;
                }
                break;

            case PLAYING:
                mPlaybackState = PlaybackState.PAUSED;
                videoPlayerView.pause();
                break;

            case IDLE:
                switch (ampCastManager.getPlaybackLocation()) {
                    case LOCAL:
                        currentPosition = 0;
                        videoPlayerContainer.prepareResource(mSelectedMedia.getContentId());
                        mPlaybackState = PlaybackState.PLAYING;
                        restartTrickplayTimer();
                        updatePlaybackLocation();
                        break;
                    case REMOTE:
                        if (ampCastManager.isConnected()) {
                            Utils.showQueuePopup(this, mPlayCircle,
                                    AmpCastUtils.createMediaQueueItem(mSelectedMedia),
                                    ampCastManager);
                        }
                        break;
                    default:
                        break;
                }
                break;
            default:
                break;
        }
        updatePlayButton(mPlaybackState);
    }

    private void stopTrickplayTimer() {
        Log.d(TAG, "Stopped TrickPlay Timer");
        if (mSeekbarTimer != null) {
            mSeekbarTimer.cancel();
        }
    }

    private void restartTrickplayTimer() {
        stopTrickplayTimer();
        mSeekbarTimer = new Timer();
        mSeekbarTimer.scheduleAtFixedRate(new UpdateSeekbarTask(), 100, 1000);
        Log.d(TAG, "Restarted TrickPlay Timer");
    }

    private void stopControllersTimer() {
        if (mControllersTimer != null) {
            mControllersTimer.cancel();
        }
    }

    private void startControllersTimer() {
        if (mControllersTimer != null) {
            mControllersTimer.cancel();
        }
        if (ampCastManager.getPlaybackLocation() == PlaybackLocation.REMOTE) {
            return;
        }
        mControllersTimer = new Timer();
        mControllersTimer.schedule(new HideControllersTask(), 5000);
    }

    // should be called from the main thread
    private void updateControllersVisibility(boolean show) {
        if (show) {
            getSupportActionBar().show();
            mControllers.setVisibility(View.VISIBLE);
        } else {
            if (!Utils.isOrientationPortrait(this)) {
                getSupportActionBar().hide();
            }
            mControllers.setVisibility(View.INVISIBLE);
        }
    }

    @Override
    protected void onPause() {
        super.onPause();
        Log.d(TAG, "onBackground() was called");
        if (ampCastManager.getPlaybackLocation() == PlaybackLocation.LOCAL) {

            if (mSeekbarTimer != null) {
                mSeekbarTimer.cancel();
                mSeekbarTimer = null;
            }
            if (mControllersTimer != null) {
                mControllersTimer.cancel();
            }
            // by calling pause before the onBackground method, the player is reset paused once return to the foreground
            if(videoPlayerView != null)
                videoPlayerView.pause();
            mPlaybackState = PlaybackState.PAUSED;
            updatePlayButton(PlaybackState.PAUSED);
        }
        ampCastManager.onBackground();
    }

    @Override
    protected void onStop() {
        Log.d(TAG, "onStop() was called");
        if(videoPlayerView != null)
            videoPlayerView.stop();
        super.onStop();
    }

    @Override
    protected void onDestroy() {
        Log.d(TAG, "onDestroy() is called");
        if(videoPlayerView != null)
            videoPlayerView.onDestroy();
        stopControllersTimer();
        stopTrickplayTimer();
        super.onDestroy();
    }

    @Override
    protected void onStart() {
        Log.d(TAG, "onStart was called");
        super.onStart();
    }

    @Override
    protected void onResume() {
        Log.d(TAG, "onForeground() was called");
        ampCastManager.onForeground();
        updatePlaybackLocation();
        if (mQueueMenuItem != null) {
            mQueueMenuItem.setVisible(ampCastManager.isConnected());
        }
        super.onResume();
    }

    @Override
    public boolean dispatchKeyEvent(@NonNull KeyEvent event) {
        return ampCastManager.getCastContext().onDispatchVolumeKeyEventBeforeJellyBean(event)
                || super.dispatchKeyEvent(event);
    }

    private class HideControllersTask extends TimerTask {

        @Override
        public void run() {
            mHandler.post(() -> {
                    updateControllersVisibility(false);
                    mControllersVisible = false;
                });

        }
    }

    private class UpdateSeekbarTask extends TimerTask {

        @Override
        public void run() {
            mHandler.post(()-> {
                    if (ampCastManager.getPlaybackLocation() == PlaybackLocation.LOCAL) {
                        int currentPos = videoPlayerView.getCurrentTimelinePosition();
                        updateSeekbar(currentPos, mDuration);
                    }
            });
        }
    }

    private void setupControlsCallbacks() {

        mSeekbar.setOnSeekBarChangeListener(new OnSeekBarChangeListener() {

            @Override
            public void onStopTrackingTouch(SeekBar seekBar) {
                if (mPlaybackState == PlaybackState.PLAYING) {
                    play(seekBar.getProgress());
                } else if (mPlaybackState != PlaybackState.IDLE) {
                    videoPlayerView.seek(seekBar.getProgress());
                }
                startControllersTimer();
            }

            @Override
            public void onStartTrackingTouch(SeekBar seekBar) {
                stopTrickplayTimer();
                stopControllersTimer();
            }

            @Override
            public void onProgressChanged(SeekBar seekBar, int progress,
                    boolean fromUser) {
                mStartText.setText(Utils.formatMillis(progress));
            }
        });

        mPlayPause.setOnClickListener(v -> {
            if (ampCastManager.getPlaybackLocation() == PlaybackLocation.LOCAL) {
                togglePlayback();
            }
        });
    }

    private void updateSeekbar(int position, int duration) {
        mSeekbar.setProgress(position);
        mSeekbar.setMax(duration);
        mStartText.setText(Utils.formatMillis(position*1000));
        mEndText.setText(Utils.formatMillis(duration*1000));
    }

    private void updatePlayButton(PlaybackState state) {
        Log.d(TAG, "Controls: PlayBackState: " + state);
        boolean isConnected = (ampCastManager.isConnected() || ampCastManager.isConnecting());
        mControllers.setVisibility(isConnected ? View.GONE : View.VISIBLE);
        mPlayCircle.setVisibility(isConnected ? View.GONE : View.VISIBLE);
        switch (state) {
            case PLAYING:
                mPlayPause.setVisibility(View.VISIBLE);
                mPlayPause.setImageDrawable(
                        getResources().getDrawable(R.drawable.ic_av_pause_dark));
                mPlayCircle.setVisibility(isConnected ? View.VISIBLE : View.GONE);
                break;
            case IDLE:
                mPlayCircle.setVisibility(View.VISIBLE);
                mControllers.setVisibility(View.GONE);
                break;
            case PAUSED:
                mPlayPause.setVisibility(View.VISIBLE);
                mPlayPause.setImageDrawable(
                        getResources().getDrawable(R.drawable.ic_av_play_dark));
                mPlayCircle.setVisibility(isConnected ? View.VISIBLE : View.GONE);
                break;
            case BUFFERING:
                mPlayPause.setVisibility(View.INVISIBLE);
                break;
            default:
                break;
        }
    }

    @SuppressLint("NewApi")
    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        getSupportActionBar().show();
        if (newConfig.orientation == Configuration.ORIENTATION_LANDSCAPE) {
            getWindow().clearFlags(WindowManager.LayoutParams.FLAG_FORCE_NOT_FULLSCREEN);
            getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
                    WindowManager.LayoutParams.FLAG_FULLSCREEN);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH) {
                getWindow().getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_LOW_PROFILE);
            }
            updateMetadata(false);
            mContainer.setBackgroundColor(getResources().getColor(R.color.black));

        } else {
            getWindow().setFlags(WindowManager.LayoutParams.FLAG_FORCE_NOT_FULLSCREEN,
                    WindowManager.LayoutParams.FLAG_FORCE_NOT_FULLSCREEN);
            getWindow().clearFlags(
                    WindowManager.LayoutParams.FLAG_FULLSCREEN);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH) {
                getWindow().getDecorView().setSystemUiVisibility(View.SYSTEM_UI_FLAG_VISIBLE);
            }
            updateMetadata(true);
            mContainer.setBackgroundColor(getResources().getColor(R.color.white));
        }
    }

    private void updateMetadata(boolean visible) {
        Point displaySize;
        if (!visible) {
            mDescriptionView.setVisibility(View.GONE);
            mTitleView.setVisibility(View.GONE);
            mAuthorView.setVisibility(View.GONE);
            displaySize = Utils.getDisplaySize(this);
            RelativeLayout.LayoutParams lp = new
                    RelativeLayout.LayoutParams(displaySize.x,
                    displaySize.y + getSupportActionBar().getHeight());
            lp.addRule(RelativeLayout.CENTER_IN_PARENT);
            videoPlayerContainer.setLayoutParams(lp);
            videoPlayerContainer.invalidate();
        } else {
            MediaMetadata mm = mSelectedMedia.getMetadata();
            mDescriptionView.setText(mSelectedMedia.getCustomData().optString(
                    VideoProvider.KEY_DESCRIPTION));
            mTitleView.setText(mm.getString(MediaMetadata.KEY_TITLE));
            mAuthorView.setText(mm.getString(MediaMetadata.KEY_SUBTITLE));
            mDescriptionView.setVisibility(View.VISIBLE);
            mTitleView.setVisibility(View.VISIBLE);
            mAuthorView.setVisibility(View.VISIBLE);
            displaySize = Utils.getDisplaySize(this);
            RelativeLayout.LayoutParams lp = new
                    RelativeLayout.LayoutParams(displaySize.x,
                    (int) (displaySize.x * mAspectRatio));
            lp.addRule(RelativeLayout.BELOW, R.id.toolbar);
            videoPlayerContainer.setLayoutParams(lp);
            videoPlayerContainer.invalidate();
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        super.onCreateOptionsMenu(menu);
        getMenuInflater().inflate(R.menu.player, menu);
        ampCastManager.initCastButton(getApplicationContext(), menu,
                R.id.media_route_menu_item);
        mQueueMenuItem = menu.findItem(R.id.action_show_queue);
        return true;
    }

    @Override
    public boolean onPrepareOptionsMenu(Menu menu) {
        menu.findItem(R.id.action_show_queue).setVisible(ampCastManager.isConnected());
        return super.onPrepareOptionsMenu(menu);
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        Intent intent;
        if (item.getItemId() == R.id.action_settings) {
            intent = new Intent(LocalPlayerActivity.this, CastPreference.class);
            startActivity(intent);
        } else if (item.getItemId() == R.id.action_show_queue) {
            intent = new Intent(LocalPlayerActivity.this, QueueListViewActivity.class);
            startActivity(intent);
        } else if (item.getItemId() == android.R.id.home) {
            ActivityCompat.finishAfterTransition(this);
        }
        return true;
    }

    private void setupActionBar() {
        Toolbar toolbar = findViewById(R.id.toolbar);
        toolbar.setTitle(mSelectedMedia.getMetadata().getString(MediaMetadata.KEY_TITLE));
        setSupportActionBar(toolbar);
        getSupportActionBar().setDisplayHomeAsUpEnabled(true);
    }

    private void loadViews() {
        videoPlayerContainer = findViewById(R.id.videoView1);
        mTitleView = findViewById(R.id.titleTextView);
        mDescriptionView = findViewById(R.id.descriptionTextView);
        mDescriptionView.setMovementMethod(new ScrollingMovementMethod());
        mAuthorView =  findViewById(R.id.authorTextView);
        mStartText =  findViewById(R.id.startText);
        mStartText.setText(Utils.formatMillis(0));
        mEndText =  findViewById(R.id.endText);
        mSeekbar = findViewById(R.id.seekBar1);
        mPlayPause = findViewById(R.id.playPauseImageView);
        mLoading =  findViewById(R.id.progressBar1);
        mControllers = findViewById(R.id.controllers);
        mContainer = findViewById(R.id.container);
        ViewCompat.setTransitionName(videoPlayerContainer, getString(R.string.transition_image));
        mPlayCircle = findViewById(R.id.play_circle);
        mPlayCircle.setOnClickListener((View v) -> togglePlayback());
        mediaRouteButton = findViewById(R.id.media_route_button);
    }
}
